# (c) Copyright 2005-2011. CodeWeavers, Inc.
package CXMenuWindows;
use warnings;
use strict;

use CXLog;
use CXUtils;
use CXMenu;

sub detect($$$$)
{
    my ($class, $cxoptions)=@_;

    my $self={
        tag         => $cxoptions->{tag},
        desktopdata => $cxoptions->{desktopdata}
    };
    bless $self, $class;
    return ($self);
}

sub id($)
{
    return "CXMenuWindows/";
}

my $wine_script;
sub get_wine_script()
{
    $wine_script=shquote_string("$ENV{CX_ROOT}/bin/wine") if (!defined $wine_script);
    return $wine_script;
}

sub get_command($;$)
{
    my ($menu, $background)=@_;

    my $lnk=CXMenu::menu_to_lnk($menu->{rawpath});

    # 'wine --start' really expects the parameter to be in the default encoding
    require CXRecode;
    $lnk=CXRecode::to_sys("UTF-8", $lnk);

    return join(" ", get_wine_script(),
                "--bottle", shquote_string($ENV{CX_BOTTLE}),
                "--check", ($background ? "--no-wait" : "--wait-children"),
                "--start", shquote_string($lnk));
}

sub delete($)
{
    my ($rawpath)=@_;

    my $wlnk=CXMenu::menu_to_lnk($rawpath);
    # 'winepath' really expects the parameter to be in the default encoding
    require CXRecode;
    $wlnk=CXRecode::to_sys("UTF-8", $wlnk);

    my $ulnk=cxbackquote(get_wine_script() . " --no-convert --wl-app winepath.exe " . shquote_string($wlnk));
    return 0 if ($? ne 0);
    chomp $ulnk;
    return 0 if (!unlink $ulnk);
    return 1;
}

sub install($$)
{
    my ($self, $components)=@_;

    my $menu=@$components[-1];
    return 1 if ($menu->{type} ne "windows");

    # The icon, if any, will be relative to the bottle
    $menu->{icon_root}||="$ENV{WINEPREFIX}/windata/cxmenu/icons";
    return 1 if ($menu->{is_dir});

    if (!defined $ENV{WINEPREFIX})
    {
        cxerr("cannot create the '$menu->{rawpath}' Windows menu because no bottle was specified\n");
        return -1; # Fatal error
    }
    if ($menu->{command} ne "")
    {
        cxwarn("'windows' menus cannot have a command. Overriding the command\n");
    }

    # Remove the parent directory from the garbage collection list
    my $path=mangle_path($menu->{rawpath});
    delete $self->{garbage_collect}->{cxdirname($path)};

    my $script="$self->{desktopdata}/cxmenu/$path";
    cxlog("Creating '$script'\n");

    my $scriptdir=cxdirname($script);
    if (!-d $scriptdir and !cxmkpath($scriptdir))
    {
        cxerr("unable to create '$scriptdir': $@\n");
        return -1; # Fatal error
    }

    if (open(my $fh, ">", $script))
    {
        print $fh "#!/bin/sh\n";
        print $fh "exec ", get_command($menu),  " \"\$\@\"\n";
        close($fh);
    }
    else
    {
        cxerr("unable to open '$script' for writing: $!\n");
        return -1; # Fatal error
    }
    chmod(0777 & ~umask(), $script);

    $menu->{command}=shquote_string($script);
    $menu->{genericname}="Windows Application (CrossOver)";
    return 1;
}

sub query($$)
{
    # Don't report the CXMenuWindows install status
    return ("", "");
}

sub uninstall($$)
{
    my ($self, $components)=@_;

    my $menu=@$components[-1];
    return 1 if ($menu->{is_dir});
    return 1 if ($menu->{type} ne "windows" and !defined $ENV{WINEPREFIX});
    # Do the uninstall for non-Windows menus so we clean up after them
    # in case their type changed

    if (!defined $ENV{WINEPREFIX})
    {
        cxerr("cannot remove the script associated to the '$menu->{rawpath}' Windows menu because no bottle was specified\n");
        return 0;
    }

    my $path=mangle_path($menu->{rawpath});
    my $script="$self->{desktopdata}/cxmenu/$path";
    if (-f $script)
    {
        cxlog("Deleting '$script'\n");
        if (!unlink $script)
        {
            cxwarn("unable to delete '$script': $!\n");
        }
    }

    # Also mark the parent directory for deletion
    $self->{garbage_collect}->{cxdirname($path)}=1;

    return 1;
}

sub removeall($$)
{
    my ($self, $pattern)=@_;

    if (defined $ENV{WINEPREFIX} and $pattern ne "legacy" and
        ($self->{tag} || "") =~ /^$pattern/)
    {
        if (opendir(my $dh, "$self->{desktopdata}/cxmenu"))
        {
            foreach my $dentry (readdir $dh)
            {
                next if ($dentry !~ /^(?:Desktop|StartMenu)/);
                my $dir="$self->{desktopdata}/cxmenu/$dentry";
                next if (!-d $dir);

                cxlog("Deleting the '$dir' directory\n");
                require File::Path;
                if (!File::Path::rmtree($dir))
                {
                    cxerr("unable to delete the '$dir' directory: $!\n");
                }
            }
            closedir($dh);
        }
        CXUtils::garbage_collect_subdirs($self->{desktopdata}, "/cxmenu", 1);
    }
    return 1;
}

sub finalize($)
{
    my ($self)=@_;
    return 1 if (!defined $ENV{WINEPREFIX});

    my $root="$self->{desktopdata}/cxmenu";
    foreach my $path (sort { $b cmp $a } keys %{$self->{garbage_collect}})
    {
        CXUtils::garbage_collect_subdirs($root, $path, 1);
    }
    CXUtils::garbage_collect_subdirs($self->{desktopdata}, "/cxmenu", 1);
    return 1;
}

return 1;
