# (c) Copyright 2005-2008, 2010. CodeWeavers, Inc.
# Mandrake uses the Debian menuing system to create the KDE .desktop files
# for KDE associations.  Other desktop systems ignore these entries.
package CXAssocMandrake;
use warnings;
use strict;
use CXAssoc;
use base "CXAssoc";
use CXLog;
use CXUtils;
use CXMimeKDE;


sub read_debian_file($)
{
    my ($self)=@_;

    return 1 if (defined $self->{filename});
    $self->{filename}="$self->{menu}/cxassoc-$self->{tag}";
    if ($self->{old_menu})
    {
        my $old_filename="$self->{old_menu}/cxassoc-$self->{tag}";
        if (-f $old_filename and !-f $self->{filename})
        {
            if (!cxmv($old_filename, $self->{filename}))
            {
                cxwarn("unable to rename '$old_filename' to '$self->{filename}': $!\n");
            }
            # Delete the parent directory if it's empty
            rmdir $self->{old_menu};
        }
    }
    $self->{lines}=[];
    $self->{menus}={};
    $self->{modified}=0;
    return 1 if (!-e $self->{filename});

    cxlog("Reading '$self->{filename}'\n");
    my $fh;
    if (!open($fh, "<", $self->{filename}))
    {
        cxerr("unable to open '$self->{filename}' for reading: $!\n");
        return 0;
    }

    my $count=0;
    foreach my $line (<$fh>)
    {
        chomp $line;
        push @{$self->{lines}}, $line;
        #cxlog(" line=[$line]\n");
        if ($line =~ /(?:\s|:)kde_filename=\"cxassoc-$self->{tag}-((?:[^\\\"]*|\\.)*)\"(?:\s|$)/)
        {
            my $massocid=unescape_string($1);
            $self->{assocs}->{$massocid}=$count;
        }
        $count++;
    }
    close($fh);
    # For debug...
    #map { cxlog("  $_\n"); } keys %{$self->{assocs}};

    return 1;
}

#####
#
# Association desktop file helper functions
#
#####

sub create_association($$$$)
{
    my ($self, $massoc, $adata, $extensions)=@_;
    $self->read_debian_file();

    CXAssoc::setup_from_best_eassoc($massoc);
    my $name=$massoc->{appname};
    if ($massoc->{verb})
    {
        CXAssoc::compute_verb_name($massoc);
        if ($massoc->{verbname})
        {
            # With the verb name coming last, its placement is wrong for
            # defining a keyboard shortcut.
            $name.=" (" . CXAssoc::remove_accelerators($massoc->{verbname}) . ")";
        }
    }
    if ($ENV{CX_TAGALL})
    {
        my $creator=$self->id();
        $creator =~ s%/%-%g;
        $name.=" ($creator)";
    }
    # We must quadruple backslashes!
    $name =~ s%\\%\\\\\\\\%g;
    # But double quotes are escaped normally
    $name =~ s%\"%\\\"%g;
    # Double all ampersands except the first one
    $name =~ s/&/&&/g;
    $name =~ s/&&/&/;
    # Double dollars
    $name =~ s/\$/\$\$/g;

    my $preference=($massoc->{mode} eq "default") ? 10 : 1;

    # Note:
    # - We may need to double percent characters in the command, but
    #   because we use a wrapper script for all associations and the name
    #   of that wrapper script has been mangled we don't need to worry about
    #   it much.
    # - We avoid conflicts between associations with identical application
    #   names (the 'title' field must be unique within a section) by using
    #   the MAssoc id to create a unique 'section' name for each association.
    my $str="?package(local." . CXUtils::get_product_id() .
            "): needs=\"kde\" kde_filename=\"cxassoc-$self->{tag}-$massoc->{id}\" section=\".hidden/$massoc->{id}\" title=\"$name\" icon=\"$massoc->{icon}\" command=\"" .
            escape_string($massoc->{command}) .
            "\" kde_opt=\"\\\\nTerminal=false\\\\nMimeType=" .
            join(";", sort keys %{$adata->{all_mimes}}) .
            "\\\\nInitialPreference=$preference\\\\nNoDisplay=true\"";

    my $line=$self->{assocs}->{$massoc->{id}};
    if (defined $line)
    {
        $self->{lines}->[$line]=$str;
    }
    else
    {
        $self->{assocs}->{$massoc->{id}}=@{$self->{lines}};
        push @{$self->{lines}}, $str;
    }
    $self->{modified}=1;

    return 1;
}

sub query_association($$$$)
{
    my ($self, $massoc, $adata, $state)=@_;
    $self->read_debian_file();

    my $line=$self->{assocs}->{$massoc->{id}};
    return $state if (!defined $line);
    my $str=$self->{lines}->[$line];

    my $cmd="";
    if ($str =~ /(?:\s|:)command=\"((?:[^\\\"]*|\\.)*)\"(?:\s|$)/)
    {
        $cmd=unescape_string($1);
    }
    return $state if ($cmd !~ /\Q$massoc->{cmdbase}\E(?:[^:]|$)/);

    my $preference="0";
    if ($str =~ /(?:\\\\n|\")InitialPreference=([0-9]+)(?:\\\\n|\")/)
    {
        $preference=$1;
    }
    return ($preference !~ /^\d+$/ or $preference < 10) ? "alternative" : "default";
}

sub delete_association($$$)
{
    my ($self, $massoc)=@_;
    $self->read_debian_file();

    my $line=$self->{assocs}->{$massoc->{id}};
    if (defined $line)
    {
        cxlog(" removing line $line\n");
        $self->{lines}->[$line]=undef;
        delete $self->{assocs}->{$massoc->{id}};
        $self->{modified}=1;
    }
    return 1;
}



#####
#
# Main
#
#####

sub detect($$$$)
{
    my ($class, $cxoptions, $cxconfig, $gui_info)=@_;
    return () if (!$gui_info->{kde_on} or
                  !$gui_info->{debian_menu_on});

    my $self={
        tag            => $cxoptions->{tag},
        menu           => $gui_info->{debian_preferred_menu},
        old_menu       => $gui_info->{debian_old_preferred_menu},
        mimes          => CXMimeKDE->get("mdk", $cxoptions, $gui_info),
        case_sensitive => 1,
        do_assoc       => 1,
        domain         => "mdk"
    };
    bless $self, $class;
    return ($self);
}

sub id($)
{
    my ($self)=@_;
    my $id="CXAssocMandrake/$self->{menu}";
    $id =~ s%/+%/%g;
    return $id;
}

sub preinstall($$)
{
    my ($self, $massoc)=@_;
    return $self->{mimes}->collect_unix_extensions($massoc);
}

sub install($$)
{
    my ($self, $massoc)=@_;
    return $self->action($self->{mimes}, $massoc, "install");
}

sub query($$)
{
    my ($self, $massoc)=@_;

    if (!$massoc)
    {
        my $id=$self->id();
        return {default     => $id,
                alternative => $id,
                mime        => $id,
                partial     => $id};
    }
    return $self->action($self->{mimes}, $massoc, "query");
}

sub uninstall($$)
{
    my ($self, $massoc)=@_;
    return $self->action($self->{mimes}, $massoc, "uninstall");
}

sub removeall($$)
{
    my ($self, $pattern)=@_;

    $self->{mimes}->removeall($self->{domain}, $pattern);

    if ($pattern eq "legacy")
    {
        $pattern="^" . CXUtils::get_product_id() . "\$";
    }
    else
    {
        $pattern="^cxassoc-$pattern";
    }

    # Scan the association database for our associations
    if (CXUtils::delete_files($self->{menu}, $pattern) > 0)
    {
        # Delete the parent directory if it's empty
        rmdir $self->{menu};
        $self->{update}=1;
    }
    if ($self->{old_menu} and CXUtils::delete_files($self->{old_menu}, $pattern) > 0)
    {
        # Delete the parent directory if it's empty
        rmdir $self->{old_menu};
        $self->{update}=1;
    }


    return 1;
}

sub finalize($)
{
    my ($self)=@_;
    my $rc=$self->{mimes}->finalize();
    if ($self->{modified})
    {
        my $empty=1;
        foreach my $line (@{$self->{lines}})
        {
            if (defined $line)
            {
                $empty=0;
                last;
            }
        }
        if ($empty)
        {
            cxlog("Deleting empty '$self->{filename}' file\n");
            # Delete the association file
            if (-e $self->{filename} and !unlink $self->{filename})
            {
                cxerr("unable to delete '$self->{filename}': $!\n");
                return 0;
            }
            # Delete the parent directory if it's empty
            rmdir $self->{menu};
        }
        else
        {
            # Save the menu file
            if (!-d $self->{menu} and !cxmkpath($self->{menu}))
            {
                cxerr("unable to create the '$self->{menu}' directory: $@\n");
                return 0;
            }
            my $fh;
            if (!open($fh, ">", $self->{filename}))
            {
                cxerr("unable to open '$self->{filename}' for writing: $!\n");
                return 0;
            }
            cxlog("Writing to '$self->{filename}'\n");
            foreach my $line (@{$self->{lines}})
            {
                next if (!defined $line);
                print $fh "$line\n";
            }
            close($fh);
        }
        $self->{update}=1;
    }
    if ($self->{update})
    {
        # Make sure we only run one instance at a time (just to be safe).
        # See also CXMenuDebian.pm.
        my $lock=CXUtils::cxlock("update-menus");
        cxsystem("update-menus");
        CXUtils::cxunlock($lock);
    }
    return $rc;
}

return 1;
