# (c) Copyright 2005-2010. CodeWeavers, Inc.
package CXAssocKDE;
use warnings;
use strict;
use CXAssoc;
use base "CXAssoc";

use CXLog;
use CXUtils;
use CXMimeKDE;



#####
#
# Association desktop file helper functions
#
#####

sub create_association($$$$)
{
    my ($self, $massoc, $adata, $extensions)=@_;

    my $dir="$self->{menu}/.hidden";
    if (!-d $dir and !cxmkpath($dir))
    {
        cxerr("unable to create the '$self->{menu}' directory: $@\n");
        return 0;
    }

    require CXRWConfig;
    my $desktop=CXRWConfig->new(undef, "dollar", "");
    my $filename="$dir/cxassoc-$self->{tag}:$massoc->{id}.desktop";
    cxlog("Creating '$filename'\n");
    $desktop->set_filename($filename);
    my $section=$desktop->append_section("Desktop Entry");
    $section->set("Encoding", "UTF-8");
    $section->set("Type", "Application");
    $section->set("X-Created-By", $self->{tag});

    CXAssoc::setup_from_best_eassoc($massoc);
    $section->set("Icon", $massoc->{icon});

    my $name=$massoc->{appname};
    if ($massoc->{verb})
    {
        CXAssoc::compute_verb_name($massoc);
        if ($massoc->{verbname})
        {
            # With the verb name coming last, its placement is wrong for
            # defining a keyboard shortcut. Plus KDE just shows them anyway.
            $name.=" (" . CXAssoc::remove_accelerators($massoc->{verbname}) . ")";
        }
    }
    $name.=" (" . $self->id() . ")" if ($ENV{CX_TAGALL});
    $section->set("Name", $name);
    if ($massoc->{localize} or $massoc->{stdverbname})
    {
        my $oldlang=CXUtils::cxgetlang();
        my $oldencoding=CXUtils::cxsetencoding("UTF-8");
        my $std_verb_names=CXAssoc::std_verb_names();
        foreach my $locale (CXUtils::get_supported_locales())
        {
            CXUtils::cxsetlang($locale);
            my $appname=$massoc->{appname};
            $appname=cxgettext($appname) if ($massoc->{localize});
            my $verbname=$std_verb_names->{$massoc->{verb}};
            $verbname=cxgettext($verbname) if ($massoc->{stdverbname});
            if ($appname ne $massoc->{appname} or $verbname ne $std_verb_names->{$massoc->{verb}})
            {
                my $name="$appname (" . CXAssoc::remove_accelerators($verbname) . ")";
                $name.=" (" . $self->id() . ")" if ($ENV{CX_TAGALL});
                $section->set("Name[$locale]", $name);
            }
        }
        CXUtils::cxsetlang($oldlang);
        CXUtils::cxsetencoding($oldencoding);
    }
    $section->set("GenericName", $massoc->{genericname}) if ($massoc->{genericname});
    if ($massoc->{genericname} and $massoc->{localize})
    {
        my $oldlang=CXUtils::cxgetlang();
        my $oldencoding=CXUtils::cxsetencoding("UTF-8");
        foreach my $locale (CXUtils::get_supported_locales())
        {
            CXUtils::cxsetlang($locale);
            my $genericname=cxgettext($massoc->{genericname});
            if ($genericname ne $massoc->{genericname})
            {
                $section->set("GenericName[$locale]", $genericname);
            }
        }
        CXUtils::cxsetlang($oldlang);
        CXUtils::cxsetencoding($oldencoding);
    }

    # The desktop file format specifies that percents must be doubled.
    my $exec=$massoc->{command};
    $exec =~ s/%/%%/g;
    $section->set("Exec", $exec);
    $section->set("Terminal", "false");
    $section->set("ServiceTypes", "false");
    my $preference=($massoc->{mode} eq "default") ? 10 : 1;
    $section->set("InitialPreference", $preference);
    $section->set("MimeType", join(";", sort keys %{$adata->{all_mimes}}));

    if (!$desktop->save())
    {
        cxerr("unable to save '$filename': $!\n");
        return 0;
    }
    return 1;
}

sub query_association($$$$)
{
    my ($self, $massoc, $adata, $state)=@_;

    my $filename="$self->{menu}/.hidden/cxassoc-$self->{tag}:$massoc->{id}.desktop";
    return $state if (!-f $filename);

    require CXRWConfig;
    my $desktop=CXRWConfig->new($filename, "dollar", "");
    cxlog("Checking '$filename'\n");

    my $section=$desktop->get_section("Desktop Entry");
    return $state if (!$section);

    my $str=$section->get("MimeType", "");
    my @mimes=split /;+/, $str;
    return $state if (CXAssoc::compare_sets(\@mimes, $adata->{all_mimes}));

    # Dedouble percents before we check the command
    $str=$section->get("Exec", "");
    $str =~ s/%%/%/g;
    return $state if ($str !~ /\Q$massoc->{cmdbase}\E(?:[^:]|$)/);

    $str=$section->get("InitialPreference", "0");
    return ($str !~ /^\d+$/ or $str < 10) ? "alternative" : "default";
}

sub delete_association($$$)
{
    my ($self, $massoc)=@_;

    require CXRWConfig;
    my $filename="$self->{menu}/.hidden/cxassoc-$self->{tag}:$massoc->{id}.desktop";
    CXRWConfig::uncache_file($filename);
    if (-f $filename)
    {
        cxlog("Deleting '$filename'\n");
        if (!unlink $filename)
        {
            cxerr("unable to delete '$filename': $!\n");
            return 0;
        }
    }
    return 1;
}



#####
#
# Main
#
#####

sub new($$$)
{
    my ($class, $cxoptions, $gui_info)=@_;
    my $self={
        tag            => $cxoptions->{tag},
        mimes          => CXMimeKDE->get("kde", $cxoptions, $gui_info),
        case_sensitive => 1,
        do_assoc       => 1,
        domain         => "kde"
    };
    bless $self, $class;
    return $self;
}

sub detect($$$$)
{
    my ($class, $cxoptions, $cxconfig, $gui_info)=@_;
    return () if (!$gui_info->{kde_on} or
                  !$gui_info->{kde_preferred_mime});

    my @selves;
    if ($gui_info->{kde_preferred_menu})
    {
        my $self=new($class, $cxoptions, $gui_info);
        $self->{menu}=$gui_info->{kde_preferred_menu};
        push @selves, $self;
    }
    if ($gui_info->{kde_preferred_alt_menu})
    {
        my $self=new($class, $cxoptions, $gui_info);
        $self->{menu}=$gui_info->{kde_preferred_alt_menu};
        push @selves, $self;
    }

    return @selves;
}

sub id($)
{
    my ($self)=@_;
    my $id="CXAssocKDE/$self->{menu}";
    $id =~ s%/+%/%g;
    return $id;
}

sub preinstall($$)
{
    my ($self, $massoc)=@_;
    return $self->{mimes}->collect_unix_extensions($massoc);
}

sub install($$)
{
    my ($self, $massoc)=@_;
    return $self->action($self->{mimes}, $massoc, "install");
}

sub query($$)
{
    my ($self, $massoc)=@_;

    if (!$massoc)
    {
        my $id=$self->id();
        return {default     => $id,
                alternative => $id,
                mime        => $id,
                partial     => $id};
    }
    return $self->action($self->{mimes}, $massoc, "query");
}

sub uninstall($$)
{
    my ($self, $massoc)=@_;
    return $self->action($self->{mimes}, $massoc, "uninstall");
}

sub removeall($$)
{
    my ($self, $pattern)=@_;

    $self->{mimes}->removeall($self->{domain}, $pattern);

    my $assoc_dir=$self->{menu};
    if ($pattern eq "legacy")
    {
        $pattern="^" . CXUtils::get_product_id() . "-app-\\d{5}\\.desktop\$";
        $assoc_dir .= "/CrossOver";
    }
    else
    {
        $pattern="$pattern\[^:]*" if ($pattern !~ s/\$$//);
        $pattern="^cxassoc-$pattern:.*\\.desktop\$";

        # We now put the desktop files in the '.hidden' subdirectory
        # but older versions were putting them in the parent.
        CXUtils::delete_files("$assoc_dir/.hidden", $pattern);
    }

    CXUtils::delete_files($assoc_dir, $pattern);

    return 1;
}

sub finalize($)
{
    my ($self)=@_;
    return $self->{mimes}->finalize();
}

return 1;
